#ident "$Header: mem.c,v 1.5 90/10/18 16:11:25 jayk Exp $"
	
/* ------------------------------------------------------------------ */
/* | Copyright Unpublished, MIPS Computer Systems, Inc.  All Rights | */
/* | Reserved.  This software contains proprietary and confidential | */
/* | information of MIPS and its suppliers.  Use, disclosure or     | */
/* | reproduction is prohibited without the prior express written   | */
/* | consent of MIPS.                                               | */
/* ------------------------------------------------------------------ */

/* mem.BIT								*/
/*									*/
/* This program sizes the available user memory, malloc's most of it	*/
/* and performs the Moving Inversions algorithm memory test.		*/
/*									*/
/* Malloc is limited by the MAXUMEM kernel parameter, so it may not be	*/
/* able to honor a request for most of free memory - so the program	*/
/* downsizes the request until it succeeds.  - This behavior does not	*/
/* indicate an error - it is normal on machines with lots of memory.	*/
/* An easy solution is to run multiple tests.  The kernel parameter	*/
/* "tune.t_maxumem" could also be increased to allow a large malloc.	*/
/*									*/
/*
 *  Moving Inversions memory test
 *
 *  The moving inversions algorithm will test all memory in roughly
 *  2 * A * N or Order(Nlog10N) where:
 *
 *	N is the number of of memory locations to be tested.
 *	A is the number of significant address bits in N
 *
 *  This algorithm can detect all stuck-at faults, all coupling faults, and
 *  some pattern sensitivety problems that can be modeled as coupling between
 *  rows and columns of the memory array.  It has the nice side effect of
 *  giving the address decoding logic a real workout.  
 *
 *  The basic algorithm reads the location under test, writes it, then verifies
 *  it.  It uses as a sliding one or zero pattern.  Memory is processed in
 *  jumps of log2(n).  One drawback of this algorithm is that it is sentive to
 *  RAM chip organization. However this is a property of all of the faster
 *  pattern sensitivity tests.  This implemenr=tation assumes a by 1 RAM.
 *
 *  INPUTS:  first_address - address of the first memory location to be tested
 *           last_address - address of the last 32 bit word to be tested.
 */

#include <sys/types.h>
#include <sys/param.h>
#include <sys/immu.h>
#include <sys/sysinfo.h>
#include <sys/sysmacros.h>
#include <sys/var.h>
#include <nlist.h>
#include <stdio.h>
	
extern int	errno;		/* system error number */
extern char	*sys_errlist[]; /* system error message strings */

#define	SINFO		1
#define	MINFO		2
#define	NNLIST		11

/* It appears that everyone is happier if these numbers are multiples
   of the page size. */
#define MINMEM		1048576	/* minimum memory to test - 2^20 */
#define SPAREMEM	3145728	/* try to leave some mem to spare - 3MB */
#define	DECMEM		1048576	/* reduce expectations on fail - 2^20 */

/* Global variables. */
char	filename[30];
FILE	*errorfile, *passfile, *sizefile;
int	processID;
int	Kmem;
float	Tdiff;

struct	minfo Mi;
struct	minfo Omi;
struct	sysinfo Osi;
struct	sysinfo Si;

struct nlist Kinfo[] = {
    { "X"		},
    { "sysinfo"	},
    { "minfo"	},
    { "X"		},
    { "X"		},
    { "X"		},
    { "X"		},
    { "X"		},
    { "X"		},
    { "X"		},
    { "X"		},
    { "X"		},
    { "X"		},
    { "X"		},
    { "X"		},
    { 0		},
};

int	forced_size;		/* optional - override bytes to test */

usage(argv)
   char *argv[];
{
    fprintf(stderr, "   usage:  %s [<bytes>] [free]\n", argv[0]);
    exit(1);
}


/* Opens unique errorfile for error messages.  If successful, then	*/
/* loops forever.							*/
main( argc, argv )
   int	argc;
   char	*argv[];
{
    register u_int *error_addr;
    register u_int msw;
    register int lsw;
    register int pattern;
    register int inverse_pattern;
    register u_int delta;
    register u_int max_delta;
    register u_int local_actual_data;
    register u_int *first_addr;
    register u_int *last_addr;
    
    int	avail;
    int	pages;
    int	counter;
    int	error;
    int	*start;
    int	dummy1;
    char step;
    u_int	pass;
    u_int	try;
    int	i;
    
    forced_size = 0;
    
    for (i = 1; i < argc; i++) {
	if (strcmp(argv[i], "free") == 0 ||
	    strcmp(argv[i], "FREE") == 0) {
	    errorfile = stderr;
	    passfile = stderr;
	    printf("%d bytes of memory free.\n", memfree());
	    exit(0);
	}
	else {
	    forced_size = atoi( argv[i] );
	    if (forced_size == 0) {
		usage(argv);
	    }
	}
    }
    
    
    processID = getpid();
    sprintf( filename, "%d.ERRmem", processID );
    errorfile = fopen( filename, "w" );
    if ( errorfile == NULL ) {
	printf( "mem.BIT: UNEXPECTED DIFFICULTY OPENING ERRORFILE!" );
	perror( "mem.BIT: UNEXPECTED DIFFICULTY OPENING ERRORFILE!" );
	exit( 1 );
    };
    
    sprintf( filename, "%d.PASSmem", processID );
    passfile = fopen( filename, "w" );
    if ( passfile == NULL ) {
	printf( "mem.BIT: UNEXPECTED DIFFICULTY OPENING PASSFILE!" );
	perror( "mem.BIT: UNEXPECTED DIFFICULTY OPENING PASSFILE!" );
	exit( 1 );
    };
    
    avail = memfree();
    
    printf("Free memory found:  %d bytes.\n", avail);
    
    if (forced_size) {
	/* Works much better if multiple of page size. */
	avail = forced_size & ~(getpagesize() - 1);
	if (avail == 0) avail = getpagesize();
	printf("Memory to test:  programmed to multiple of page size - %d bytes.\n", avail);
	
    } else if (avail > (SPAREMEM + MINMEM)) {
	avail -= SPAREMEM;
	printf("Memory to test:  %d bytes.\n", avail);
	
    } else {
	avail = MINMEM;
	printf("Memory to test:  set to minumum - %d bytes.\n", avail);
    }
    
    sprintf( filename, "%d.OPTmem", processID );
    sizefile = fopen( filename, "w" );
    
    for ( pass = 1; ; pass++ ) {
	rewind( passfile );
	fprintf( passfile, "%u\n", pass );
	fflush( passfile);
	
	for (try = 0; ; try ++) {
	    /*
	     * Find the size chunk of memory that the system will
	     * allow to be allocated.
	     */
	    if ( (start = (int *)malloc( avail )) == 0 ) {
		printf( "Couldn't malloc %d bytes on pass %d try %d - retrying...\n",
		       avail, pass, try );
		
		if ( avail > MINMEM ) {
		    avail -= DECMEM;
		}
		else {
		    printf( "Unable to allocate a significant piece of memory\n");
		    printf( "Malloc size of %d is less than minimum of %d\n",
			   avail, MINMEM );
		    fprintf( errorfile, "Unable to allocate a significant piece of memory\n");
		    fprintf( errorfile, "Malloc size of %d is less than minimum of %d\n",
			    avail, MINMEM );
		    pexit();
		}
		
		continue;
	    }
	    
	    break;
	}
	
	printf("Testing %d bytes of memory.\n", avail);
	
	if ( sizefile != NULL ) {
	    rewind( sizefile );
	    fprintf( sizefile, "%7d KB\n", avail / 1000 );
	    fflush( sizefile );
	};

	first_addr = (u_int *)start;
	last_addr = (u_int *)((int)first_addr + avail - 16);
	error = 0;
	
	/*
	 * Initialize all memory to zeroes.
	 */
	for ( error_addr = first_addr; error_addr < last_addr; ) {
	    *error_addr++ = 0;
	}

	if (avail >= 9994240 && pass < 10) { /* 10MB rounded to a page */
	    step = 'a';
	    rewind( passfile );
	    fprintf( passfile, "%u%c\n", pass, step );
	    fflush( passfile);
	}

	pattern = 0;
	inverse_pattern = ~pattern;
	
	/*
	 * Compute the number of bits required to represent the highest
	 * address then turn it into a bit mask.
	 */
	max_delta =  1 << ((4 * 8) - 1);
	for ( msw = (((char *)last_addr - (char *)first_addr) + 1) << 1; (int) msw > 0; msw <<= 1 ) {
	    max_delta >>= 1;
	}
	
	/*
	 * Address incrementing loop.
	 */
	for ( delta = 4; delta != max_delta; delta <<= 1 ) {
	    /*
	     * Forward moving inversions test.
	     */
	    do {
		for ( lsw = 0; lsw <= (delta - 1); lsw += 4 ) {
		    for (msw = (u_int)first_addr; 
			 msw <= ((u_int)last_addr & ~(delta - 1));
			 msw = ((msw + delta) < msw ? (int)last_addr + 1 : msw + delta) ) {
			if ((msw + lsw) > (int)last_addr) break;
			error_addr = ((u_int *)(msw + lsw));
			if ( (local_actual_data = *error_addr) != pattern ) {
			    error = 1;
			    goto done;
			}
			
			*error_addr = inverse_pattern;
			if ( (local_actual_data = *error_addr) != inverse_pattern ) {
			    error = 2;
			    goto done;
			}
		    }
		}
		
		pattern = inverse_pattern;
		inverse_pattern = ~pattern;
	    } while ( pattern );
	    
	    /*
	     * Reverse moving inversions.
	     */
	    do {
		for ( lsw = delta - 4; lsw >= 0; lsw -= 4 ) {
		    for (msw = (u_int)(last_addr + 1) - delta; 
			 msw >= (u_int)first_addr;
			 msw = delta > msw ? 0 : msw - delta ) {
			if ((msw + lsw) > (int)last_addr) break;
			error_addr = (u_int *)(msw + lsw);
			if ( (local_actual_data = *error_addr) != pattern ) {
			    error = 1;
			    goto done;
			}
			
			*error_addr = inverse_pattern;
			if ( (local_actual_data = *error_addr) != inverse_pattern ) {
			    error = 2;
			    goto done;
			}
		    }
		}
		
		pattern = inverse_pattern;
		inverse_pattern = ~pattern;
	    } while ( pattern );

	    if (avail >= 9994240 && pass < 10) { /* 10MB rounded to a page */
		step++;
		rewind( passfile );
		fprintf( passfile, "%u%c\n", pass, step );
		fflush( passfile);
	    }
	}
    done:
	if ( error == 1 ) {
	    printf( "Address %08x: Expect %08x, Actual %08x, Xor %08x\n",
		   error_addr, pattern, local_actual_data, pattern ^ local_actual_data );
	    fprintf( errorfile, "Address %08x: Expect %08x, Actual %08x, Xor %08x\n",
		    error_addr, pattern, local_actual_data, pattern ^ local_actual_data );
	}
	else if ( error == 2 ) {
	    printf( "Address %08x: Expect %08x, Actual %08x, Xor %08x\n",
		   error_addr, inverse_pattern, local_actual_data, inverse_pattern ^ local_actual_data );
	    fprintf( errorfile, "Address %08x: Expect %08x, Actual %08x, Xor %08x\n",
		    error_addr, inverse_pattern, local_actual_data, inverse_pattern ^ local_actual_data );
	}
	
	free( start );
    }
};


getkern()
{
    int	i;
    
    if ( (Kmem = open( "/dev/kmem", 0 )) == -1 ) {
	printf( "Unable to open /dev/kmem.\n");
	printf( "Check that mem.BIT owner is root and mode is 4755.\n");
	fprintf( errorfile, "Unable to open /dev/kmem.\n");
	fprintf( errorfile, "Check that mem.BIT owner is root and mode is 4755.\n");
	pexit();
    }
    
    if ( nlist( "/unix", Kinfo ) == -1 ) {
	printf( "Unable to extract name list from /unix.");
	fprintf( errorfile, "Unable to extract name list from /unix.");
	pexit();
    }
    
    for ( i = 0; i < NNLIST; i++ ) {
	Kinfo[ i ].n_value &= ~(0x80000000);
    }
    
}


getvals()
{
    int	i;
    int	j;
    int	k;
    
    lseek( Kmem, (long)Kinfo[ SINFO ].n_value, 0 );
    if ( read( Kmem, &Si, sizeof( Si ) ) == -1 ) {
	printf( "Unable to read system info from /dev/kmem.\n");
	
	fprintf( errorfile, "Unable to read system info from /dev/kmem.\n");
	pexit();
    }
    
    if ( Kinfo[ MINFO ].n_value != 0 ) {
	lseek( Kmem, (long)Kinfo[ MINFO ].n_value, 0 );
	if ( read( Kmem, &Mi, sizeof( Mi ) ) == -1 ) {
	    printf( "Unable to read memory info from /dev/kmem.\n");
	    fprintf( errorfile, "Unable to read memory info from /dev/kmem.\n");
	    pexit();
	}
    }
}


pexit()
{
    printf( "errno %d - %s\n", errno, sys_errlist[errno]);
    fprintf( errorfile, "errno %d - %s\n", errno, sys_errlist[errno]);
    exit( 1 );
}


/* Return the number of bytes of free memory. */
int memfree()
{
    double	magic = 4.294967296e9;
    double	tmp;
    int	avail;
    u_long	m0;
    u_long	m1;
    
    getkern();
    getvals();
    
    Osi = Si;
    Omi = Mi;
    
    sleep(1);
    getvals();
    
    Tdiff = Si.cpu[ 0 ] - Osi.cpu[ 0 ] + Si.cpu[ 1 ] - Osi.cpu[ 1 ]
	+ Si.cpu[ 2 ] - Osi.cpu[ 2 ] + Si.cpu[ 3 ] - Osi.cpu[ 3 ];
    
    m1 = Mi.freemem[ 1 ] - Omi.freemem[ 1 ];
    if ( Mi.freemem[ 0 ] >= Omi.freemem[ 0 ] ) {
	m0 = Mi.freemem[ 0 ] - Omi.freemem[ 0 ];
    }
    else {
	m0 = m1 + (~(Omi.freemem[ 0 ] - Mi.freemem[ 0 ]));
	--m1;
    }
    
    tmp = (double)((m0 + magic * m1) / Tdiff);
    
    avail = (int)tmp * getpagesize();
    
    return(avail);
}
