#ident "$Header: mport.c,v 1.1 90/03/13 14:27:59 jayk Exp $"

/* ------------------------------------------------------------------ */
/* | Copyright Unpublished, MIPS Computer Systems, Inc.  All Rights | */
/* | Reserved.  This software contains proprietary and confidential | */
/* | information of MIPS and its suppliers.  Use, disclosure or     | */
/* | reproduction is prohibited without the prior express written   | */
/* | consent of MIPS.                                               | */
/* ------------------------------------------------------------------ */

/* This program opens the tty ports on a system and then generates	*/
/* random strings which it writes to and reads from the ports through	*/
/* jumpers installed by the operator.					*/
/*									*/
/* The gettys must be turned off on the ports.				*/

#include <sys/types.h>

/* For timeout error during read (int receive). */
#include <signal.h>

/* Flag values for calls to "open". */
#include <fcntl.h>

/* Used for file manipulation. */
#include <stdio.h>

/* Used to read error returns from UNIX system calls. */
#include <errno.h>
extern char *sys_errlist[];

/* Used for tty setup and control. */
#include <sys/termio.h>
#include <sys/stermio.h>

/* Used to return system time stamp for seeding random number generator. */
#include <time.h>

/* Used to make program read better. */
#define NO_ERROR	-1000
#define BAD		1
#define GOOD		0
#define ZERO		0
#define TEST		50

#define MINBAUD		1200
#define MAXBAUD		19200
#define	MAXPORT		3
#define MINLENGTH	5
#define MAXLENGTH	250

#define	TIME		30
#define	DELAY		20000

#define	SPACE		0x20
#define	TILDA		0x7e

extern int	timeout();

/* Global variables. */
int	errno;
int	alrm;
int	gotalarm;

int	port;
char	*p1arr[] = {
#ifdef	MODEM
	"m1",
	"m2",
	"m3",
#else	MODEM
	"1",
	"2",
	"3",
#endif	MODEM
};					/* to set the port variable */

int	curbaud;
int	numtoread;
int	numread;
int	numports;
int	writereadport;
int     MaxBaud;

FILE	*errorfile;
FILE	*passfile;

struct	termio cb;

char	writestring[ MINLENGTH + MAXLENGTH ];
char	readstring[ MINLENGTH + MAXLENGTH ];

/* Opens the error file, checks the arguments, seeds the random number	*/
/* generator, and starts the tty test.					*/
main( argc, argv )
int	argc;
char	*argv[];
{
	long	time();

	openfiles();
	checkargs( argc, argv );
	srand( time( 0 ) );
	testport();
}

/* Opens the errorfile and passfile.  The filename contains the process	*/
/* ID of the current invocation of this program, which makes it easy to	*/
/* kill later.								*/
openfiles()
{
	int	processID;
	char	buffer[ 30 ];

	processID = getpid();
	sprintf( buffer, "%d.ERRmprt", processID );
	errorfile = fopen( buffer, "w" );
	if ( errorfile == NULL ) {
		printf( "port.BIT: UNEXPECTED DIFFICULTY OPENING ERRORFILE!" );
		exit( 1 );
	}

	sprintf( buffer, "%d.PASSmprt", processID );
	passfile = fopen( buffer, "w" );
	if ( passfile == NULL ) {
		printf( "port.BIT: UNEXPECTED DIFFICULTY OPENING PASSFILE!" );
		exit( 1 );
	}
}

/* Checks whether the arguments to the program are legitimate. */
checkargs( argc, argv )
int	argc;
char	*argv[];
{
	if ( argc < 2 ) {
		fprintf( errorfile, "   usage: port.BIT <number_of_ports>\n" );
		fclose( errorfile );
		exit( 1 );
	}

	numports = atoi( argv[ 1 ] );
	if ( (numports <= 0) || (numports > MAXPORT) ) {
		fprintf( errorfile,
		    "   port.BIT: numports must be > 0 and <= %d\n", MAXPORT );
		fclose( errorfile );
		exit( 1 );
	}

	if ( argc > 2 ) {
		/* arg2 is max baud */
		MaxBaud = atoi( argv[ 2 ] );
	}
	else {
		MaxBaud = MAXBAUD;
	}
}

/* This is the function that actually tests the ICP's.  It does so in	*/
/* an infinite loop, so the program must be killed to exit.  Pairs of	*/
/* ports are opened and then tested at various baud rates.		*/
testport()
{
	int	j, i;			/* counter for for loops */
	int	result;
	u_int	pass;

	curbaud = MINBAUD;
	for ( pass = 1; ; pass++ ) {
		rewind( passfile );
		fprintf( passfile, "%u\n", pass );
		fflush( passfile );
		for ( j = 0; j < numports; j++ ) {
			port = j;
			do {
				openport();
				setstrings();
				result = transmit();
				if ( result == GOOD ) {
					result = receive();
					if ( result == GOOD ) {
						compare();
					}

				}

				closeport();
				bumpbaud();
			} while ( curbaud != MINBAUD );
		}
	}
}

/* Tries to open a pair of ports.  Generates error message if		*/
/* difficulty encountered.  If tries all ports unsuccessfully, aborts	*/
/* program.								*/
openport()
{
	int	result;

	printf( "CURRENT BAUD RATE IS %d\n", curbaud );
	result = openreadwrite();
	if ( result == BAD ) {
		fprintf( errorfile,
		    "   EXCESSIVE NUMBER OF FAILURES OPENING PORT--" );
		fprintf( errorfile, "ABORTING PROGRAM\n" );
		fclose( errorfile );
		exit( 1 );
	}
}

/* Opens the port to write to and read from.  Sets up various		*/
/* parameters.  If encounters difficulty, generates error message.	*/
openreadwrite()
{
	char	buffer[ 30 ];

	errno = NO_ERROR;
	sprintf( buffer, "/dev/tty%s", p1arr[ port ] );
	printf( " CCC PORT IS /dev/tty%s\n", p1arr[ port ] );

	writereadport = open( buffer, O_RDWR | O_NDELAY );
	if ( writereadport == EOF ) {
		fprintf( errorfile, "   COULD NOT OPEN '%s'", buffer );
		fprintf( errorfile, " AS TRANSMIT PORT" );
		if ( errno != NO_ERROR ) {
			fprintf( errorfile, "--UNIX ERROR RETURN: %d", errno );
		}

		fprintf( errorfile, "\n\n" );
		return( BAD );
	}
	else {
		if ( (errno = ioctl( writereadport, TCGETA, &cb )) == -1 ) {
			fprintf( errorfile, "--ioctl 1 openreadwrite()--" );
			if ( errno != NO_ERROR ) {
				fprintf( errorfile, "--UNIX ERROR RETURN: %d",
				    errno );
			}

			fprintf( errorfile, "\n\n" );
			return( BAD );
		}

		cb.c_cc[ VTIME ] = 0;
		cb.c_cc[ VMIN ] = 1;
		cb.c_lflag &= ~(ISIG | ICANON | ECHO);
		cb.c_cflag &= ~(PARENB | CBAUD | CSIZE | HUPCL);
		cb.c_cflag |= CS7 | CREAD;
		switch ( curbaud ) {
		case  1200:
			cb.c_cflag |= B1200;
			break;

		case  2400:
			cb.c_cflag |= B2400;
			break;

		case  4800:
			cb.c_cflag |= B4800;
			break;

		case  9600:
			cb.c_cflag |= B9600;
			break;

		case 19200:
			cb.c_cflag |= EXTA;
			break;
		}

/*
		cb.c_iflag &= ~(INPCK | ISTRIP | ICRNL | IXON | IXANY | IXOFF);
		cb.c_iflag |= IGNBRK | IGNPAR;
		cb.c_oflag &= ~OPOST;
*/
		cb.c_iflag = 0;
		cb.c_oflag &= ~OPOST;

		if ( (errno = ioctl( writereadport, TCSETAW, &cb )) == -1 ) {
			fprintf( errorfile, "--ioctl 3 openreadwrite()--" );
			if ( errno != NO_ERROR ) {
				fprintf( errorfile, "--UNIX ERROR RETURN: %d",
				    errno );
			}

			fprintf( errorfile, "\n\n" );
			return( BAD );
		}
	}

	return( GOOD );
}

/* First pads both readstring and writestring with nulls, then, in	*/
/* order to insure that the ports are tested with a wide variety of bit	*/
/* patterns, this function generates a random string between MINLENGTH	*/
/* and (MINLENGTH + MAXLENGTH - 1) characters in length.  The string	*/
/* can contain any character from 1-255, so it is not necessarily	*/
/* printable (error messages print it out in hex).			*/
setstrings()
{
	int	index;
	int	length;

	index = 0;
	while ( index < (MINLENGTH + MAXLENGTH) ) {
		writestring[ index ] = '\0';
		readstring[ index ] = '\0';
		index++;
	}

	index = 0;
	length = (rand() % MAXLENGTH) + MINLENGTH;
	/* length = TEST; */

	/* Use just printable ASCII characters */
	do {
		writestring[ index ] = rand() % TILDA;
		if ( writestring[ index ] <= SPACE ) {
			writestring[ index ] += SPACE;
		}

		index++;
	} while ( index < length );
}

/* Sends the current random string to the writereadport.  Looks for	*/
/* UNIX error returns and generates an error message if detects any.	*/
transmit()
{
	int	delay;
	int	n;
	int	numtowrite;
	int	numwritten;
	int	*trash;

	errno = NO_ERROR;
	ioctl( writereadport, TCFLSH, 1 );
	if ( (errno = ioctl( writereadport, TCFLSH, 0 )) == -1 ) {
		fprintf( errorfile, "--ioctl FAILED in transmit()" );
		fprintf( errorfile, "\n\n" );
		if ( errno != NO_ERROR ) {
			fprintf( errorfile, "--UNIX ERROR RETURN: %d", errno );
			fprintf( errorfile, "\n\n" );
			return( BAD );
		}
	}

	numtowrite = strlen( writestring );
	numwritten = 0;
	for ( n = 0; n < numtowrite; n++ ) {
		numwritten += write( writereadport, &writestring[ n ], 1 );
		for ( delay = 0; delay < DELAY; delay++ ) {
			;
		}
	}

	printf( " CCC NUMTOWRITE = %d  NUMWRITTEN = %d \n",
	    numtowrite, numwritten );
	if ( numwritten != numtowrite ) {
		fprintf( errorfile, "   TRANSMIT ERROR:\n" );
		printsets();
		fprintf( errorfile, "     TRIED TO TRANSMIT (IN HEX):\n" );
		printstring( writestring, 8 );
		fprintf( errorfile, "     CHARACTERS TRANSMITTED: %d\n",
		    numwritten );
		if ( errno != NO_ERROR ) {
			fprintf( errorfile, "     UNIX ERROR RETURN     : %d\n",
			    errno );
		}

		fprintf( errorfile, "\n" );
		return( BAD );
	}
	else {
		return( GOOD );
	}
}

/* Prints current settings. */
printsets()
{
	fprintf( errorfile, "     PORT: '/dev/tty" );
	fprintf( errorfile, "%s'\n", p1arr[ port ] );

	fprintf( errorfile, "     CURRENT BAUD RATE: %d\n", curbaud );
}

/* Prints the given string in hex.  If the string is longer than a	*/
/* single line, it breaks it up onto more than one line.		*/
printstring( thistring, indent )
char	thistring[];
int	indent;
{
	int	index;
	int	counter;
	int	maxindex;

	counter = 0;
	while ( counter < indent ) {
		fprintf( errorfile, " " );
		counter++;
	}

	fprintf( errorfile, "'" );
	index = 0;
	maxindex = strlen( thistring );
	while ( index < maxindex ) {
		fprintf( errorfile, "%2.2x", thistring[ index ] & 0xff );
		index++;
		if ( ((index % 35) == 0) && (index < maxindex) ) {
			fprintf( errorfile, "\n" );
			counter = 0;
			while ( counter <= indent ) {
				fprintf( errorfile, " " );
				counter++;
			}
		}
	}

	fprintf( errorfile, "'\n" );
}

#define MAXTIME 60

int	curtime;

timeout()
{
	fprintf( errorfile,
	    "AN ALARM SIGNAL OCCURRED WHILE ATTEMPTING TO READ A TTY PORT AFTER %d SECONDS\n",
	    curtime );
	fflush( errorfile );
	gotalarm = 1;
}

/* Reads a random string from the writereadport.  Looks for UNIX error	*/
/* returns and generates an error message if detects any.		*/
receive()
{
	int	cnt1, cnt2, flags;

	errno = NO_ERROR;
	signal( SIGALRM, timeout );
	flags  = fcntl( writereadport, F_GETFL, 0 );
	flags &= ~O_NDELAY;
	fcntl( writereadport, F_SETFL, flags );
	alrm = GOOD;
	cnt2 = numtoread = strlen( writestring );

	curtime = TIME;
	gotalarm = numread = 0;
	alarm( TIME );
	while ( !gotalarm && ((cnt1 = read( writereadport, &readstring[ numread ], cnt2 )) != cnt2 )) {
		cnt2 -= cnt1;
		numread += cnt1;
	}

	if ( cnt1 == cnt2 ) {
		numread += cnt1;	/* to catch last read when cnt1 = cnt2 */
	}

	alarm( 0 );
	if ( (numread != numtoread) || gotalarm ) {
		printsets();
		fprintf( errorfile, "READMISS " );
		if ( cnt1 == -1 ) {
			fprintf( errorfile, "ERRNO='%s'; ", sys_errlist[ errno ] );
		}

		if ( numread > 0 ) {
			fprintf(errorfile, "READ %d OUT OF %d CHARS; ", 
			    numread, numtoread);
		}

		if ( gotalarm ) {
			/* non-block read */
			flags  = fcntl( writereadport, F_GETFL, 0 );

			flags |= O_NDELAY;
			fcntl( writereadport, F_SETFL, flags );

			cnt1 = read( writereadport, readstring, numtoread-numread );
			if ( cnt1 > 0 ) {
				fprintf( errorfile, "RETRY GOT %d CHARS", cnt1 );
			}
			else if ( cnt1 != 0 ) {
				fprintf( errorfile, "NONBLOCKING READ ERROR:'%s'", 
				    sys_errlist[ errno ]);
			}

			flags  = fcntl( writereadport, F_GETFL, 0 );
			flags &= ~O_NDELAY;
			fcntl( writereadport, F_SETFL, flags );

		}
		else if ( cnt1 != -1 ) {
			fprintf(errorfile, "; READ ANOMALY");
		}

		fprintf( errorfile, "\n" );
		fflush( errorfile );
		return( BAD );
	}

	return( GOOD );
}

/* Compares the string sent and the string received.  Generates an	*/
/* error if not identical.						*/
compare()
{
	if ( (strcmp( writestring, readstring ) != 0) && (alrm == GOOD) ) {
		fprintf( errorfile, "   COMPARE ERROR:\n" );
		printsets();
		fprintf( errorfile, "     STRING TRANSMITTED (IN HEX):\n" );
		printstring( writestring, 8 );
		fprintf( errorfile, "     STRING RECEIVED (IN HEX):\n" );
		printstring( readstring, 8 );
		fprintf( errorfile, "     CHARACTERS TRANSMITTED: " );
		fprintf( errorfile, "%d\n", strlen( writestring ) );
		fprintf( errorfile, "     CHARACTERS RECEIVED   : " );
		fprintf( errorfile, "%d\n\n", strlen( readstring ) );
	}
}

/* Increments the curbaud baud rate.  If baud rate exceeds MAXBAUD,	*/
/* resets it to MINBAUD.						*/
bumpbaud()
{
	curbaud = curbaud * 2;
	if ( curbaud > MaxBaud ) {
		curbaud = MINBAUD;
	}
}

/* Closes the current pair of ports.  Generates an error message if	*/
/* encounters difficulty doing so.					*/
closeport()
{
	errno = NO_ERROR;
	if ( close( writereadport ) != 0 ) {
		fprintf( errorfile, "   ERROR CLOSING PORT '/dev/tty" );
		fprintf( errorfile, "%s'", p1arr[ port ] );

		if ( errno != NO_ERROR ) {
			fprintf( errorfile, "--UNIX ERROR RETURN: %d", errno );
		}

		fprintf( errorfile, "\n\n" );
	}
}
