#ident "$Header: port.c,v 1.5 91/05/14 13:37:24 jayk Locked $"
    
/* ------------------------------------------------------------------ */
/* | Copyright Unpublished, MIPS Computer Systems, Inc.  All Rights | */
/* | Reserved.  This software contains proprietary and confidential | */
/* | information of MIPS and its suppliers.  Use, disclosure or     | */
/* | reproduction is prohibited without the prior express written   | */
/* | consent of MIPS.                                               | */
/* ------------------------------------------------------------------ */
    
/* This program opens the tty ports on a system and then generates	*/
/* random strings which it writes to and reads from the ports through	*/
/* jumpers installed by the operator.					*/
/*									*/
/* The gettys must be turned off on the ports.				*/
    
#include <sys/types.h>
    
/* For timeout error during read (int receive). */
#include <signal.h>
    
/* Flag values for calls to "open". */
#include <fcntl.h>
    
/* Used for file manipulation. */
#include <stdio.h>
    
/* Used to read error returns from UNIX system calls. */
#include <errno.h>
extern char *sys_errlist[];

/* Used for tty setup and control. */
#include <sys/termio.h>
#include <sys/stermio.h>

/* Used to return system time stamp for seeding random number generator. */
#include <time.h>

/* Function declarations for strings. */
#include <string.h>
#include <ctype.h>

/* Used to make program read better. */
#define NO_ERROR	-1000
#define BAD		1
#define GOOD		0
#define	FAST		1
#define	SLOW		0
#define TEST		50

#define XMITLOW		0
#define XMITHIGH	1
#define MINBAUD		1200
#define MAXBAUD		19200
#define MINLENGTH	5
#define MAXLENGTH	250
#define MAXPORT		64

#define	DELAY		20000

#define	TIME		120

#define	SPACE		0x20
#define	TILDA		0x7e

extern int	timeout();

/* Global variables. */
int	errno;
int	alrm;
int	gotalarm;
int	burst;

int	port1;
int	port2;
char	**ports;

/* Port names for the various asynchronous serial controllers. */

/* Digiboard COM/8 and COM/16 (or PC/16) AT controlers */
char	*digi[] = {
    "d0",  "d1",  "d2",  "d3",  "d4",  "d5",  "d6",  "d7",
    "d8",  "d9",  "d10", "d11", "d12", "d13", "d14", "d15",
    "d16", "d17", "d18", "d19", "d20", "d21", "d22", "d23",
    "d24", "d25", "d26", "d27", "d28", "d29", "d30", "d31",
    "d32", "d33", "d34", "d35", "d36", "d37", "d38", "d39",
    "d40", "d41", "d42", "d43", "d44", "d45", "d46", "d47",
    "d48", "d49", "d50", "d51", "d52", "d53", "d54", "d55",
    "d56", "d57", "d58", "d59", "d60", "d61", "d62", "d63",
};

/* Mizar VME-ICP16/X VME controller */
char	*icp[] = {
    "h0",  "h1",  "h2",  "h3",  "h4",  "h5",  "h6",  "h7",
    "h8",  "h9",  "h10", "h11", "h12", "h13", "h14", "h15",
    "i0",  "i1",  "i2",  "i3",  "i4",  "i5",  "i6",  "i7",
    "i8",  "i9",  "i10", "i11", "i12", "i13", "i14", "i15",
    "j0",  "j1",  "j2",  "j3",  "j4",  "j5",  "j6",  "j7",
    "j8",  "j9",  "j10", "j11", "j12", "j13", "j14", "j15",
    "k0",  "k1",  "k2",  "k3",  "k4",  "k5",  "k6",  "k7",
    "k8",  "k9",  "k10", "k11", "k12", "k13", "k14", "k15",
};

/* Systech 16-port VME controller - modem ports use "ma0"... */
char	*systech[] = {
    "a0",  "a1",  "a2",  "a3",  "a4",  "a5",  "a6",  "a7",
    "a8",  "a9",  "a10", "a11", "a12", "a13", "a14", "a15",
    "a16", "a17", "a18", "a19", "a20", "a21", "a22", "a23",
    "a24", "a25", "a26", "a27", "a28", "a29", "a30", "a31",
    "a32", "a33", "a34", "a35", "a36", "a37", "a38", "a39",
    "a40", "a41", "a42", "a43", "a44", "a45", "a46", "a47",
    "a48", "a49", "a50", "a51", "a52", "a53", "a54", "a55",
    "a56", "a57", "a58", "a59", "a60", "a61", "a62", "a63",
};

int	curtime;
int	curmode;
int	curbaud;
int	numtoread;
int	numread;
int	writeport;
int	readport;
int	firstport = 1;		/* numbering starts at 1 */
int	numports = 0;		/* this must be even */
int	maxbaud = MAXBAUD;	/* determines baud rates to test */
int	maxport = MAXPORT;	/* limits how many ports are supported */
char	controller[30] = "";	/* type of controller board */
int	debug = 0;		/* print out extra info when set to 1 */

FILE	*errorfile;
FILE	*passfile;
char	filename[30];

struct	termio cb;

char	writestring[ MINLENGTH + MAXLENGTH ];
char	readstring[ MINLENGTH + MAXLENGTH ];

/* Opens the error file, checks the arguments, seeds the random number	*/
/* generator, and starts the tty test.					*/
main( argc, argv )
   int	argc;
   char	*argv[];
{
    long	time();
    
    checkargs( argc, argv );
    openfiles();
    setup();
    srand( time( 0 ) );
    testports();
}

/* Opens the errorfile and passfile.  The filename contains the process	*/
/* ID of the current invocation of this program, which makes it easy to	*/
/* kill later.								*/
openfiles()
{
    int	processID;
    char	buffer[ 30 ];
    
    processID = getpid();
    sprintf( buffer, "%d.ERRprt", processID );
    errorfile = fopen( buffer, "w" );
    if ( errorfile == NULL ) {
	fprintf(stderr,"port.BIT: UNEXPECTED DIFFICULTY OPENING ERRORFILE!" );
	exit( 1 );
    }
    
    sprintf( filename, "%d.PASSprt", processID );
    passfile = fopen( filename, "w" );
    if ( passfile == NULL ) {
	fprintf(stderr,"port.BIT: UNEXPECTED DIFFICULTY OPENING PASSFILE!" );
	exit( 1 );
    }
}

usage_lines()
{
    fprintf(stderr,"\nusage: port.BIT <ports> [-c <ctrl>] [-b <maxbaud>] [-debug]\n\n");
    fprintf(stderr,"       <ports> = number of ports (must be even) or\n");
    fprintf(stderr,"                 range of ports.  ie) 3-4, 5-16, 2-5.\n");
    fprintf(stderr,"       -c <ctrl> = Type of controller.  ie) digi, icp, systech.\n");
    fprintf(stderr,"       -b <maxbaud> = Maximum baud rate. Default is 19200.\n");
    fprintf(stderr,"       -debug = Display debug info and set burst mode.\n");

    exit( 1 );
}


/* Checks whether the arguments to the program are legitimate. */
checkargs( argc, argv )
   int	argc;
   char	*argv[];
{
    int arg, lastport;
    char *sptr;

    if ( argc < 2 ) {
	usage_lines();
    }
    
    if ((sptr = strchr(argv[1], '-')) != NULL) {
	/* This is a range of ports. */
	firstport = atoi( argv[1] );
	lastport = atoi( ++sptr );
	if (firstport <= 0 || firstport > MAXPORT-1 ||
	    lastport <= 1 || lastport > MAXPORT) {
	    fprintf(stderr,"port.BIT: invalid range specification: firstport=%d lastport=%d.\n",
		   firstport, lastport);
	    usage_lines();
	}
	numports = lastport - firstport + 1;
	if ( numports % 2 || numports < 2 ) {
	    fprintf(stderr,"port.BIT: number of ports (%d) must even and greater than 0.\n", 
		   numports);
	    usage_lines();
	}
    } else {
	/* This is the number of ports. */
	numports = atoi( argv[1] );
	if ( numports > MAXPORT ) {
	    fprintf(stderr,"port.BIT: no more than %d ports are supported.\n",MAXPORT);
	    usage_lines();
	}

	if ( numports % 2 ) {
	    fprintf(stderr,"port.BIT: the number of ports must be even.\n" );
	    usage_lines();
	}
    }
    
    for (arg=2; arg<argc; arg++) {

	if (strncmp("-c", argv[arg], 2) == 0) {
	    if (++arg >= argc) {
		fprintf(stderr,"port.BIT: missing argument for -c option.\n");
		usage_lines();
	    }
	    strcpy(controller, argv[arg]);
	    for (sptr=controller; *sptr; sptr++) 
		if (isupper(*sptr)) *sptr = tolower(*sptr);
	    if (strncmp("digi", controller, 4) &&
		strncmp("icp", controller, 3) &&
		strncmp("systech", controller, 7)) {
		fprintf(stderr,"port.BIT: invalid controller for -c option - %s.\n",
		       controller);
		fprintf(stderr,"   try: digi, icp, systech, etc.\n");
		usage_lines();
	    }

	} else if (strncmp("-b", argv[arg], 2) == 0) {
	    if (++arg >= argc) {
		fprintf(stderr,"port.BIT: missing argument for -b option.\n");
		usage_lines();
	    }
	    maxbaud = atoi( argv[arg] );
	    if (1200 != maxbaud &&
		2400 != maxbaud &&	
		4800 != maxbaud &&
		9600 != maxbaud &&
		19200 != maxbaud) {
		fprintf(stderr,"port.BIT: invalid maximum baud rate for -b option - %d.\n",
		       maxbaud);
		fprintf(stderr,"   try: 1200, 2400, 4800, 9600, 19200, etc.\n");
		usage_lines();
	    }

	} else if (strncmp("-d", argv[arg], 2) == 0) {
	    debug = 1;

	} else {
	    fprintf(stderr,"port.BIT: unrecognized argument - %s.\n", argv[arg]);
	    usage_lines();
	}

    } /* for (arg=... */
}

/* Setup the program based on the arguments. */
setup()
{
    /* Make sure that the user selected a controller that exists. */
    if (*controller) {
	if (! strncmp("icp", controller, 3)) {
	    ports = icp;
	} else if (! strncmp("digi", controller, 4)) {
	    ports = digi;
	} else if (! strncmp("systech", controller, 7)) {
	    ports = systech;
	} else {
	    fprintf(stderr,"port.BIT: unrecognized controller = %s\n", controller);
	    fprintf(errorfile,"port.BIT: unrecognized controller = %s\n", controller);
	    exit(1);
	}	

	if ( ! devexists(ports[firstport])) {
	    fprintf(stderr,"port.BIT: /dev/tty%s cannot be opened - %s controller not accessible.\n", 
		   ports[firstport], controller);
	    fprintf(errorfile,"port.BIT: /dev/tty%s cannot be opened - %s controller not accessible.\n", 
		   ports[firstport], controller);
	    exit(1);
	}

    /* If user didn't select a controller, find one. */
    } else { 
	if (devexists(icp[firstport])) {
	    ports = icp;
	} else if (devexists(digi[firstport])) {
	    ports = digi;
	} else if (devexists(systech[firstport])) {
	    ports = systech;
	} else {
	    fprintf(stderr,"port.BIT: no serial controller found.\n");
	    fprintf(stderr,"   Can't open /dev/tty%s, /dev/tty%s, or /dev/tty%s.\n",
		   icp[firstport], digi[firstport], systech[firstport]);
	    fprintf(errorfile,"port.BIT: no serial controller found.\n");
	    fprintf(errorfile,"   Can't open /dev/tty%s, /dev/tty%s, or /dev/tty%s.\n",
		   icp[firstport], digi[firstport], systech[firstport]);
	    exit(1);
	}
    }	

    /* VME based systems */
    if (ports == icp) {
	burst = FAST;

    /* AT based systems */
    } else {
	burst = SLOW;
    }

    
    if (debug) {
	burst = FAST;
	printf("\nDebug mode - display termio info for each tty.\n");
	printf("           - Set burst mode to FAST (normally just VME).\n");
	printf("ports      = %d to %d\n", firstport, firstport+numports-1);
	printf("controller = %s\n", controller);
	printf("baud rates = %d to %d\n", MINBAUD, maxbaud);
	printf("burst mode = %d\n", burst);
    };
}

printtime()
{
    long time();
    char *ctime();
    long seconds;
    
    seconds = time(0);
    fprintf( errorfile, "     %s\n", ctime(&seconds) );
    fflush( errorfile );
}

/* This is the function that actually tests the ports.  It does so in	*/
/* an infinite loop, so the program must be killed to exit.  Pairs of	*/
/* ports are opened and then tested at various baud rates.  After each	*/
/* pairs have been tested, the global variable curmode is changed so	*/
/* that whichever member of each pair was the one that transmitted	*/
/* becomes the one that receives, and vice versa.			*/
/*									*/
/* NOTE: All ports are initialized by writing one character to it prior	*/
/*	 to the actual test.  Right after bootup of the system, this	*/
/*	 test times out on the first read only.  It looks like some	*/
/*	 system initialization is solved by accessing the ports first.	*/
testports()
{
    int	j, i;			/* counter for for loops */
    int	result;
    u_int	pass;
    
    curbaud = MINBAUD;
    curmode = XMITLOW;
    for ( port1 = firstport-1; port1 < firstport + numports - 1; port1++ ) {
	if ( openwrite() ) {
	    fprintf( errorfile,
		    "   FAILURE INITIALIZING PORT--" );
	    fprintf( errorfile, "ABORTING PROGRAM\n" );
	    printtime();
	    fclose( errorfile );
	    exit( 1 );
	}
	
	write( writeport, "x", 1 );
	sleep( 1 );
	close( writeport );
    }
    
    sleep( 1 );
    
    for ( pass = 1; ; pass++ ) {
	
	rewind( passfile );
	fprintf( passfile, "%u\n", pass );
	fflush( passfile);
	
	for ( j = firstport-1; j < (firstport + numports - 1); j += 2 ) {
	    port1 = j;
	    port2 = port1 + 1;
	    curmode = XMITLOW;
	    do {
		do {
		    openpair();
		    setstrings();
		    result = transmit();
		    if ( result == GOOD ) {
			result = receive();
			if ( result == GOOD ) {
			    result = compare();
			}
		    }
		    
		    closepair();
		    bumpbaud();
		} while (curbaud != MINBAUD);
		
		flipmode();
		sleep( 1 );
	    } while ( curmode == XMITHIGH );
	}
    }
}

/* Tries to open a pair of ports.  Generates error message if		*/
/* difficulty encountered.  If tries all ports unsuccessfully, aborts	*/
/* program.								*/
openpair()
{
    int	result;
    
    printf( "CURRENT BAUD RATE IS %d\n", curbaud );
    result = openwrite();
    if ( result == GOOD ) {
	result = openread();
    }
    
    if ( result == BAD ) {
	fprintf( errorfile,
		"   EXCESSIVE NUMBER OF FAILURES OPENING PORTS--" );
	fprintf( errorfile, "ABORTING PROGRAM\n" );
	printtime();
	fclose( errorfile );
	exit( 1 );
    }
}

/* Tries to open the device file to determine if it exists.            */
/* Returns 1 if it exists, 0 if it doesn't.                            */
int
devexists(dev)
   char *dev;
{
    char buffer[30];
    int filedesc;

    sprintf(buffer, "/dev/tty%s", dev);
    filedesc = open(buffer, O_RDWR | O_NDELAY);

    if (filedesc >= 0) {
	return(1);
    } else {
	errno = NO_ERROR;
	return(0);
    }
}

/* Opens the port to write to.  Sets up various parameters.  If		*/
/* encounters difficulty, generates error message.  Which value it uses	*/
/* for the tty number depends on curmode.				*/
openwrite()
{
    char	buffer[ 30 ];
    
    errno = NO_ERROR;
    if ( curmode == XMITLOW ) {
	sprintf( buffer, "/dev/tty%s", ports[ port1 ] );
	printf( " CCC WRITEPORT IS /dev/tty%s\n", ports[ port1 ] );
    }
    else {
	sprintf( buffer, "/dev/tty%s", ports[ port2 ] );
	printf( " CCC WRITEPORT IS /dev/tty%s\n", ports[ port2 ] );
    }
    
    writeport = open( buffer, O_WRONLY | O_NDELAY );
    if ( writeport == EOF ) {
	fprintf( errorfile, "   COULD NOT OPEN '%s'", buffer );
	fprintf( errorfile, " AS TRANSMIT PORT\n" );
	if ( errno != NO_ERROR ) {
	    fprintf( errorfile, "   UNIX ERRNO %d: '%s'\n",
		    errno, sys_errlist[ errno ] );
	}
	fprintf( errorfile, "\n\n" );
	printtime();
	return( BAD );
    }
    else {
	if ( ioctl( writeport, TCGETA, &cb ) == -1 ) {
	    fprintf( errorfile, "--ioctl 1 openwrite()--\n" );
	    if ( errno != NO_ERROR ) {
		fprintf( errorfile, "UNIX ERRNO %d: '%s'\n",
			errno, sys_errlist[ errno ] );
	    }
	    
	    fprintf( errorfile, "\n" );
	    printtime();
	    return( BAD );
	}
	
	
	if (debug) printf("	termio(old): i=%07o  o=%07o  c=%07o  l=%07o\n", 
			  cb.c_iflag, cb.c_oflag, cb.c_cflag, cb.c_lflag);
	
	cb.c_cc[ VTIME ] = 1;
	cb.c_cc[ VMIN ] = 1;
	cb.c_iflag = BRKINT | INPCK | IXON | IXOFF;
	cb.c_oflag = 0;
	cb.c_cflag = CS7 | CREAD | PARENB | HUPCL | CLOCAL;
	cb.c_lflag = 0;
	
	switch ( curbaud ) {
	  case  1200: cb.c_cflag |= B1200; break;
	  case  2400: cb.c_cflag |= B2400; break;
	  case  4800: cb.c_cflag |= B4800; break;
	  case  9600: cb.c_cflag |= B9600; break;
	  case 19200: cb.c_cflag |= EXTA;  break;
	}
	
	if (debug) printf("	termio(new): i=%07o  o=%07o  c=%07o  l=%07o\n", 
			  cb.c_iflag, cb.c_oflag, cb.c_cflag, cb.c_lflag);
	
	if ( ioctl( writeport, TCSETAW, &cb ) == -1 ) {
	    fprintf( errorfile, "--ioctl 3 openwrite()--\n" );
	    if ( errno != NO_ERROR ) {
		fprintf( errorfile, "UNIX ERRNO %d: '%s'\n",
			errno, sys_errlist[ errno ] );
	    }
	    
	    fprintf( errorfile, "\n" );
	    printtime();
	    return( BAD );
	}
	
    }
    return( GOOD );
}

/* Opens the port to read from.  Sets up various parameters.  If	*/
/* encounters difficulty, generates error message.  Which value it uses	*/
/* for the tty number depends on curmode.				*/
openread()
{
    char	buffer[ 30 ];
    int flags;
    
    errno = NO_ERROR;
    if ( curmode == XMITLOW ) {
	sprintf( buffer, "/dev/tty%s", ports[ port2 ] );
	printf( " CCC READPORT IS /dev/tty%s\n", ports[ port2 ] );
    }
    else {
	sprintf( buffer, "/dev/tty%s", ports[ port1 ] );
	printf( " CCC READPORT IS /dev/tty%s\n", ports[ port1 ] );
    }
    
    readport = open( buffer, O_RDONLY | O_NDELAY );
    if ( readport == EOF ) {
	fprintf( errorfile, "   COULD NOT OPEN '%s'", buffer );
	fprintf( errorfile, " AS RECEIVE PORT\n" );
	if ( errno != NO_ERROR ) {
	    fprintf( errorfile, "   UNIX ERRNO %d: '%s'\n",
		    errno, sys_errlist[ errno ] );
	}
	
	fprintf( errorfile, "\n" );
	printtime();
	return( BAD );
    }
    else {
	flags = fcntl( readport, F_GETFL, 0 );
	flags &= ~O_NDELAY;
	fcntl( readport, F_SETFL, flags );
	if ( ioctl( readport, TCGETA, &cb ) == -1 ) {
	    fprintf( errorfile, "--ioctl 2 openread()--\n" );
	    if ( errno != NO_ERROR ) {
		fprintf( errorfile, "UNIX ERRNO %d: '%s'\n",
			errno, sys_errlist[ errno ] );
	    }
	    
	    fprintf( errorfile, "\n" );
	    printtime();
	    return( BAD );
	}
	
	if (debug) printf("	termio(old): i=%07o  o=%07o  c=%07o  l=%07o\n", 
			  cb.c_iflag, cb.c_oflag, cb.c_cflag, cb.c_lflag);
	
	cb.c_cc[ VTIME ] = 1;
	cb.c_cc[ VMIN ] = 1;
	cb.c_iflag = BRKINT | INPCK | IXON | IXOFF;
	cb.c_oflag = 0;
	cb.c_cflag = CS7 | CREAD | PARENB | HUPCL | CLOCAL;
	cb.c_lflag = 0;
	
	switch ( curbaud ) {
	case  300:
	    cb.c_cflag |= B300;
	    break;
	    
	case  600:
	    cb.c_cflag |= B600;
	    break;
	    
	case  1200:
	    cb.c_cflag |= B1200;
	    break;
	    
	case  2400:
	    cb.c_cflag |= B2400;
	    break;
	    
	case  4800:
	    cb.c_cflag |= B4800;
	    break;
	    
	case  9600:
	    cb.c_cflag |= B9600;
	    break;
	    
	case 19200:
	    cb.c_cflag |= EXTA;
	    break;
	}
	
	if (debug) printf("	termio(new): i=%07o  o=%07o  c=%07o  l=%07o\n", 
			  cb.c_iflag, cb.c_oflag, cb.c_cflag, cb.c_lflag);
	
	if ( ioctl( readport, TCSETAF, &cb ) == -1 ) {
	    fprintf( errorfile, "--ioctl 4 openread()--\n" );
	    if ( errno != NO_ERROR ) {
		fprintf( errorfile, "UNIX ERRNO %d: '%s'\n",
			errno, sys_errlist[ errno ] );
	    }
	    
	    fprintf( errorfile, "\n" );
	    printtime();
	    return( BAD );
	}
    }
    
    return( GOOD );
}

/* First pads both readstring and writestring with nulls, then, in	*/
/* order to insure that the ports are tested with a wide variety of bit	*/
/* patterns, this function generates a random string between MINLENGTH	*/
/* and (MINLENGTH + MAXLENGTH - 1) characters in length.  The string	*/
/* can contain any character from 1-255, so it is not necessarily	*/
/* printable (error messages print it out in hex).			*/
setstrings()
{
    int	index;
    int	length;
    
    index = 0;
    while ( index < (MINLENGTH + MAXLENGTH) ) {
	writestring[ index ] = '\0';
	readstring[ index ] = '\0';
	index++;
    }
    
    index = 0;
    length = (rand() % MAXLENGTH) + MINLENGTH;
    /* length = TEST; */
    
    /* Use just printable ASCII characters */
    do {
	writestring[ index ] = rand() % TILDA;
	if ( writestring[ index ] <= SPACE ) {
	    writestring[ index ] += SPACE;
	}
	
	index++;
    } while ( index < length );
}

/* Sends the current random string to the writeport.  Looks for UNIX	*/
/* error returns and generates an error message if detects any.		*/
transmit()
{
    int	delay;
    int	n;
    int	numtowrite;
    int	numwritten;
    int	*trash;
    
    errno = NO_ERROR;
    ioctl( writeport, TCFLSH, 1 );
    if ( ioctl( readport, TCFLSH, 0 ) == -1 ) {
	fprintf( errorfile, "--ioctl FAILED in transmit()" );
	fprintf( errorfile, "\n\n" );
	if ( errno != NO_ERROR ) {
	    fprintf( errorfile, "UNIX ERRNO %d: '%s'",
		    errno, sys_errlist[ errno ] );
	    fprintf( errorfile, "\n\n" );
	    printtime();
	    return( BAD );
	}
    }
    
    sleep(1);
    
    numtowrite = strlen( writestring );
    if ( burst == FAST ) {
	numwritten = write( writeport, writestring, numtowrite );
    }
    else {
	numwritten = 0;
	for ( n = 0; n < numtowrite; n++ ) {
	    numwritten += write( writeport, &writestring[ n ], 1 );
	    for ( delay = 0; delay < DELAY; delay++ ) {
		;
	    }
	}
    }
    
    printf( " CCC NUMTOWRITE = %d  NUMWRITTEN = %d \n",
	   numtowrite, numwritten );
    if ( numwritten != numtowrite ) {
	fprintf( errorfile, "   TRANSMIT ERROR:" );
	printtime();
	printsets();
	fprintf( errorfile, 
		"     TRIED TO TRANSMIT (IN HEX) (%d CHARACTERS):\n",
		numwritten );
	printstring( writestring, 8 );
	if ( errno != NO_ERROR ) {
	    fprintf( errorfile, "     UNIX ERRNO %d: '%s'\n",
		    errno, sys_errlist[ errno ] );
	}
	
	fprintf( errorfile, "\n" );
	return( BAD );
    }
    else {
	return( GOOD );
    }
}

/* Prints current settings. */
printsets()
{
    if ( curmode == XMITLOW ) {
	fprintf( errorfile, "     TRANSMIT: '/dev/tty%s'",
		ports[ port1 ] );
	fprintf( errorfile, "  RECEIVE: '/dev/tty%s'",
		ports[ port2 ] );
    } else {
	fprintf( errorfile, "     TRANSMIT: '/dev/tty%s'",
		ports[ port2 ] );
	fprintf( errorfile, "  RECEIVE: '/dev/tty%s'",
		ports[ port1 ] );
    }
    fprintf( errorfile, "  BAUD RATE: %d\n", curbaud );
}

/* Prints the given string in hex.  If the string is longer than a	*/
/* single line, it breaks it up onto more than one line.		*/
printstring( thistring, indent )
   char	thistring[];
   int	indent;
{
    int	index;
    int	counter;
    int	maxindex;
    
    counter = 0;
    while ( counter < indent ) {
	fprintf( errorfile, " " );
	counter++;
    }
    
    fprintf( errorfile, "'" );
    index = 0;
    maxindex = strlen( thistring );
    while ( index < maxindex ) {
	fprintf( errorfile, "%2.2x", thistring[ index ] & 0xff );
	index++;
	if ( ((index % 35) == 0) && (index < maxindex) ) {
	    fprintf( errorfile, "\n" );
	    counter = 0;
	    while ( counter <= indent ) {
		fprintf( errorfile, " " );
		counter++;
	    }
	}
    }
    
    fprintf( errorfile, "'\n" );
}

timeout()
{
    fprintf( errorfile,
	    "AN ALARM SIGNAL OCCURRED WHILE ATTEMPTING TO READ A TTY PORT AFTER %d SECONDS\n",
	    curtime );
    fflush( errorfile );
    gotalarm = 1;
}

/* Reads a random string from the readport.  Looks for UNIX error	*/
/* returns and generates an error message if detects any.		*/
receive()
{
    int	cnt1, cnt2, flags;
    
    errno = NO_ERROR;
    signal( SIGALRM, timeout );
    flags = fcntl( readport, F_GETFL, 0 );
    flags &= ~O_NDELAY;
    fcntl( readport, F_SETFL, flags );
    alrm = GOOD;
    cnt2 = numtoread = strlen( writestring );
    
    curtime = TIME;
    gotalarm = numread = 0;
    alarm( TIME );
    while ( !gotalarm && ((cnt1 = read( readport, &readstring[ numread ], cnt2 )) != cnt2 )) {
	cnt2 -= cnt1;
	numread += cnt1;
    }
    
    if ( cnt1 == cnt2 ) {
	numread += cnt1;	/* to catch last read when cnt1 = cnt2 */
    }
    
    alarm( 0 );
    if ( (numread != numtoread) || gotalarm ) {
	printsets();
	fprintf( errorfile, "READMISS " );
	if ( cnt1 == -1 ) {
	    fprintf( errorfile, "ERRNO='%s'; ", sys_errlist[ errno ] );
	}
	
	if ( numread > 0 ) {
	    fprintf(errorfile, "READ %d OUT OF %d CHARS; ", 
		    numread, numtoread);
	}
	
	if ( gotalarm ) {
	    /* non-block read */
	    flags = fcntl( readport, F_GETFL, 0 );
	    flags |= O_NDELAY;
	    fcntl( readport, F_SETFL, flags );
	    
	    cnt1 = read( readport, readstring, numtoread-numread );
	    if ( cnt1 > 0 ) {
		fprintf( errorfile, "RETRY GOT %d CHARS", cnt1 );
	    }
	    else if ( cnt1 != 0 ) {
		fprintf( errorfile, "NONBLOCKING READ ERROR:'%s'", 
			sys_errlist[ errno ]);
	    }
	    
	    flags = fcntl( readport, F_GETFL, 0 );
	    flags &= ~O_NDELAY;
	    fcntl( readport, F_SETFL, flags );
	}
	else if ( cnt1 != -1 ) {
	    fprintf(errorfile, "; READ ANOMALY");
	}
	
	fprintf( errorfile, "\n" );
	printtime();
	return( BAD );
    }
    
    return( GOOD );
}

/* Display the number of characters that differ between two strings. */
/* If the strings differ, specify the position of the first difference. */
printdiffs(a, b)
   char *a, *b;
{
    int i, len, lena, lenb, diffs, first_diff;
    
    lena = strlen(a);
    lenb = strlen(b);
    if (lena > lenb) {	/* len = min(lena,lenb) */
	len = lenb;
	diffs = lena - lenb;
    } else {
	len = lena;
	diffs = lenb - lena;
    }
    
    first_diff = len;
    for (i=0; i<len; i++)
	if (*(a+i) != *(b+i)) {
	    diffs++;
	    if (i < first_diff) first_diff = i;
	}
    if (diffs != 0) {
	fprintf( errorfile,
		"     %d CHARACTERS DIFFER.  FIRST DIFF AT CHAR %d\n",
		diffs, first_diff+1);
    }
}

/* Compares the string sent and the string received.  Generates an	*/
/* error if not identical.						*/
compare()
{
    if ( (strcmp( writestring, readstring ) != 0) && (alrm == GOOD) ) {
	fprintf( errorfile, "   COMPARE ERROR:" );
	printtime();
	printsets();
	printdiffs( writestring, readstring );
	fprintf( errorfile, 
		"     STRING TRANSMITTED (IN HEX) (%d CHARACTERS):\n",
		strlen( writestring ) );
	printstring( writestring, 8 );
	fprintf( errorfile, 
		"     STRING RECEIVED (IN HEX) (%d CHARACTERS):\n",
		strlen( readstring ) );
	printstring( readstring, 8 );
	return(BAD);
    } else {
	return(GOOD);
    }
}

/* Increments the curbaud baud rate.  If baud rate exceeds MAXBAUD,	*/
/* resets it to MINBAUD.						*/
bumpbaud()
{
    curbaud = curbaud * 2;
    if ( curbaud > maxbaud ) {
	curbaud = MINBAUD;
    }
}

/* Closes the current pair of ports.  Generates an error message if	*/
/* encounters difficulty doing so.					*/
closepair()
{
    errno = NO_ERROR;
    if ( close( writeport ) != 0 ) {
	fprintf( errorfile, "   ERROR CLOSING TRANSMIT PORT '/dev/tty" );
	if ( curmode == XMITLOW ) {
	    fprintf( errorfile, "%s'\n", ports[ port1 ] );
	}
	else {
	    fprintf( errorfile, "%s'\n", ports[ port2 ] );
	}
	
	if ( errno != NO_ERROR ) {
	    fprintf( errorfile, "   UNIX ERRNO %d: '%s'\n",
		    errno, sys_errlist[ errno ] );
	}
	
	fprintf( errorfile, "\n" );
    }
    
    errno = NO_ERROR;
    if ( close( readport ) != 0 ) {
	fprintf( errorfile, "   ERROR CLOSING RECEIVE PORT '/dev/tty" );
	if ( curmode == XMITLOW ) {
	    fprintf( errorfile, "%s'\n", ports[ port2 ] );
	}
	else {
	    fprintf( errorfile, "%s'\n", ports[ port1 ] );
	}
	
	if ( errno != NO_ERROR ) {
	    fprintf( errorfile, "   UNIX ERRNO %d: '%s'\n",
		    errno, sys_errlist[ errno ] );
	}
	
	fprintf( errorfile, "\n" );
    }
}

/* Changes the direction of transmit and receive ports. */
flipmode()
{
    if ( curmode == XMITLOW ) {
	curmode = XMITHIGH;
    }
    else {
	curmode = XMITLOW;
    }
}


/*
This section describes how the tty's are setup.  It also shows what the
system defaults are and what BIT used to use.  port.BIT did not work
under 4.50, at first and had to be modified.

ABCDEF
Column A - Risc/OS 4.0, etc
Column B - Risc/OS 4.50, etc
Column C - BIT transmit port under 4.0, etc.
Column D - BIT receive port under 4.0, etc.
Column E - 
Column F - BIT now (for both 4.0 and 4.50)

termio.c_iflag:

---- -	IGNBRK  0000001  Ignore break condition.
-XXX X	BRKINT  0000002  Signal interrupt on break.
-XXX -	IGNPAR  0000004  Ignore characters with parity errors.
---- -	PARMRK  0000010  Mark parity errors.
---- X	INPCK   0000020  Enable input parity check.
-X-X -	ISTRIP  0000040  Strip character.
---- -	INLCR   0000100  Map NL to CR on input.
---- -	IGNCR   0000200  Ignore CR.
-X-X -	ICRNL   0000400  Map CR to NL on input.
---- -	IUCLC   0001000  Map upper-case to lower-case on input.
-X-- X	IXON    0002000  Enable start/stop output control.
-X-X -	IXANY   0004000  Enable any character to restart output.
---- X	IXOFF   0010000  Enable start/stop input control.

termio.c_oflag:

-X-- -	OPOST   0000001  Postprocess output.
---- -	OLCUC   0000002  Map lower case to upper on output.
-XX- -	ONLCR   0000004  Map NL to CR-NL on output.
---- -	OCRNL   0000010  Map CR to NL on output.
---- -	ONOCR   0000020  No CR output at column 0.
---- -	ONLRET  0000040  NL performs CR function.
---- -	OFILL   0000100  Use fill characters for delay.
---- -	OFDEL   0000200  Fill is DEL, else NUL.
---- -	NLDLY   0000400  Select new-line delays:
---- -	NL0     0
---- -	NL1     0000400
---- -	CRDLY   0003000  Select carriage-return delays:
---- -	CR0     0
---- -	CR1     0001000
---- -	CR2     0002000
---- -	CR3     0003000
---- -	TABDLY  0014000  Select horizontal-tab delays:
---- -	TAB0    0
---- -	TAB1    0004000
---- -	TAB2    0010000
-XX- -	TAB3    0014000  Expand tabs to spaces.
---- -	BSDLY   0020000  Select backspace delays:
---- -	BS0     0
---- -	BS1     0020000
---- -	VTDLY   0040000  Select vertical-tab delays:
---- -	VT0     0
---- -	VT1     0040000
---- -	FFDLY   0100000  Select form-feed delays:
---- -	FF0     0
---- -	FF1     0100000

termio.c_cflag:

---- -	CBAUD         0000017  Baud rate:
---- -	B0            0        Hang up
---- -	B50           0000001  50 baud
---- -	B75           0000002  75 baud
---- -	B110          0000003  110 baud
---- -	B134          0000004  134 baud
---- -	B150          0000005  150 baud
---- -	B200          0000006  200 baud
---- -	B300          0000007  300 baud
---- -	B600          0000010  600 baud
---- -	B1200         0000011  1200 baud
---- -	B1800         0000012  1800 baud
---- -	B2400         0000013  2400 baud
---- -	B4800         0000014  4800 baud
---- -	B9600         0000015  9600 baud
---- -	B19200        0000016 19200 baud
---- -	EXTA          0000016  External A
---- -	B38400        0000017  38400 baud
---- -	EXTB          0000017  External B
---- -	CSIZE         0000060  Character size:
---- -	CS5           0        5 bits
---- -	CS6           0000020  6 bits
--XX X	CS7           0000040  7 bits
-X-- -	CS8           0000060  8 bits
---- -	CSTOPB        0000100  Send two stop bits, else one.
-XXX X	CREAD         0000200  Enable receiver.
---- X	PARENB        0000400  Parity enable.
---- -   PARODD        0001000  Odd parity, else even.
-XX- X	HUPCL         0002000  Hang up on last close.
-X-- X	CLOCAL        0004000  Local line, else dial-up.
---- -	LOBLK         0040000  Block layer output.
---- -	CNEW_MDMBUF   0100000  Start/stop output when carrier drops.

termio.c_lflag:

-X-- -	ISIG          0000001  Enable signals.
-X-- -	ICANON        0000002  Canonical input (erase and kill processing).
---- -	XCASE         0000004  Canonical upper/lower presentation.
-X-- -	ECHO          0000010  Enable echo.
---- -	ECHOE         0000020  Echo erase character as BS-SP-BS.
-XX- -	ECHOK         0000040  Echo NL after kill character.
---- -	ECHONL        0000100  Echo NL.
---- -	NOFLSH        0000200  Disable flush after interrupt or quit.
---- -	LNEW_CTLECH   0002000  Echo control characters as ^X, delete as ^?
---- -	LNEW_PRTERA   0004000  Printing terminal erase mode
---- -	LNEW_FLUSHO   0010000  Output is being flushed
---- -	LNEW_CRTBS    0020000  Backspace on erase rather than echoing erase
---- -	LNEW_PENDIN   0040000  Retype pending input at next
---- -	read or input character
---- -	TOSTOP        0100000  Send SIGTTOU on output by a background job

*/
