/*
	FPSIGNAL.C - a default signal handler for floating point
	exceptions.  This consists of three parts:

	fpsigset() - a routine to set fpsigcatch() as the handler
	for signal SIGFPEs.  fpsigset takes two arguments, a pointer
	to a user function to be called on receipt of the signal, before
	any message printing is done, and an action flag - telling it 
	whether to abort on receipt of the signal and whether to print a 
	diagnostic message.

	fpsigcatch() - a default catcher.  fpsigcatch will:

		call the user routine specified by the call to fpsigset
		if one was specified.

		print a diagnostic message detailing the error
		unless this was turned off by the call to fpsigset.

		abort or return, depending on which was specified by
		the call to fpsigset.

	****** 	UPDATED 10/17/85 for the FORTRAN fpe stuff, and for the
		Turbo fpa option.  fpsigcatch() is still the handler for
		the SIGFPE.  If there is a FORTRAN user handler and 
		the error was emulated by software, the error is translated
		to that given by hardware, and the user handler called. 
		If the FORTRAN user handler returns, the program is 
		aborted, optionally with a core dump.

		Otherwise, if there is a C user handler, it is called. 
		
*/

#include <stdio.h>
#include <sys/signal.h>
#include <fpsignal.h>
#include <fperr.h>
#include <nan.h>
#define TRUE 1
#define FALSE 0
#include <sys/kversion.h>
#ifndef KVERS_CPUTYPE
#define KVERS_CPUTYPE 3
#endif

static int give_diagnostics=TRUE,abort_on_signal=TRUE,dump_on_abort=TRUE;
static int fperror = 0;
/*unsigned long _hw_errmask = (IGN_UNDERFLOW|IGN_DENORM|IGN_INEXACT);*/
unsigned long _hw_errmask; 
static char _has_turbo_fpa = FALSE;

static int (*_C_user_fpehandler)() = 0;
static int has_fort_user_fpehandler = FALSE;
int _is_fortran; 

unsigned long fakehwerror() ;
static int fpsigcatch();

fpsigset(handler,code) int (*handler)();
int code;

{
	/* set up the Turbo hardware, if we are using it */
	/* always ignore the inexact bit */
	code |= IGN_INEXACT ;
	_hw_errmask = (code & 0xffff0000);
	sethw((code >> 16));

	/* specify fpsigcatch as the handler */
	signal(SIGFPE,fpsigcatch);

	abort_on_signal = TRUE;

	if (code & FORT_HANDLER)
		/* cant continue after a fortran fp error */
		has_fort_user_fpehandler = 1;
	else {
		_C_user_fpehandler = handler;
		if (code & CONTINUE_AFTER_FPERROR) abort_on_signal = FALSE;
	}

	if (code & INHIBIT_FPMESSAGE) give_diagnostics = FALSE;
	else give_diagnostics = TRUE;

	if (code & INHIBIT_DUMP) dump_on_abort = FALSE;
	else dump_on_abort = TRUE;


}

static union {
		float fval;
		long float dval; } _fptemp;


static 
fpsigcatch(sig) int sig;

{

	/* in the case of a math error, the incoming value will be 
	   either an operand or a partial result.  In the former case,
	   we want to save it (for printing by fpmessage()) before calling
	   the user handler */

	
	if (_fperror.operation == MATH) _fptemp.dval = _fperror.val.dval;

	if (!_is_software) {
		/* get the hardware error code... */
		if (!_has_turbo_fpa) {
			/* this should never happen, but just in case... */	
			fperror = (IGN_UNKNOWN >> 16);
		}
		else {
			fpspl7();
			fperror = fpclear();
			/* the bits NOT set in fperror are the errors,  we
		   	   need the complement */
			fperror = (~fperror & 0xff);
			fperror <<= 16;
		}
	}

	/* call the default handler */
	if (/*(_is_fortran)&&*/(_is_software)) {
		/* translate the error to a hardware error type.
		   Call the FORTRAN handler with the result.
		*/
		fperror = fakehwerror();
		if (fperror == 0) {
			signal(SIGFPE,fpsigcatch);
			return;
		}
	}

	if ((_is_fortran)||(!_is_software)) {
		hw_to_swerror(fperror);
		if (!_is_software) _fperror.precision = 0;
	}

	if (has_fort_user_fpehandler) {
		unsigned long forterrorcode;
		if ((!_is_fortran) && (_is_software))
			/* if this error was generated by C and
			   we are about to call the fortran error
			   handler, it needs to be translated.
			*/
			fperror = fakehwerror();
		forterrorcode = (fperror >> 16);
		call_fort_user_handler(&forterrorcode);
	}
	else if (_C_user_fpehandler)  
		(*_C_user_fpehandler)();
	else
		if ((_fperror.operation != MATH) || (_fperror.type != PARTIAL_SLOSS))
			_fperror.val.dval = 0.0;

	if (give_diagnostics) fpmessage();

	if (abort_on_signal) {
	    if (dump_on_abort) {
	        signal(SIGFPE,SIG_DFL);
		kill(getpid(),SIGFPE);
	    } else kill(getpid(),SIGKILL);
	} 

	if ((!_is_software)&&(_has_turbo_fpa))
		fpspl0();

	signal(SIGFPE,fpsigcatch);
}



static char *precstr[3] = {"unknown","single","double"};

static char *opstr[NOPS+1] = {
				"<unknown operation>",
				"add",
				"subtract",
				"multiply",
				"divide",
				"float_to_integer",
				"change_precision",
				"mod",
				"compare",
				"convert",
				"math routine" };

static char *math_routine[NMATH_ROUTINES+1] = {
				"UNKNOWN",
				"sin",
				"cos",
				"tan",
				"log",
				"exp",
				"sqrt",
				"pow",
				"asin",
				"acos",
				"sinh",
				"cosh",
				"atan2",
				"atan",
				"up_i",
				"gamma",
				"hypot",
				"j0",
				"j1",
				"y0",
				"y1",
				"yn",
				"log10",
				"tanh",
				"jn"
				};


fpmessage() {

	/* print a diagnostic message pertaining to the 
	   floating point exception */

	   char *msgstr;
	   int op = _fperror.operation;

	if ((_fperror.precision != SINGLE)&&(_fperror.precision != DOUBLE)) 
		_fperror.precision = 0;
	if ((op < ADD ) || (op > NOPS))
		op = 0;
	if ((_mathfunc_id < 0) || (_mathfunc_id > NMATH_ROUTINES))
		_mathfunc_id = 0;
	fprintf(stderr,"**FLOATING POINT EXCEPTION ");
	if (_fperror.precision == UNKNOWN) {
		if (_fperror.operation != 0)
			fprintf(stderr,"on operation %s %s ",opstr[op],
				op == MATH?math_routine[_mathfunc_id]:"");
	}
	else {
		fprintf(stderr,"on %s precision ", precstr[_fperror.precision]);
		if (_fperror.operation != 0)
			fprintf(stderr,"%s %s ",opstr[op],
				op == MATH?math_routine[_mathfunc_id]:"");
	}

	fprintf(stderr,"\n\t-- ");
	switch (_fperror.type) {

		case INVALID_OP: 	msgstr = "NaN or INF as operand";
							break;

		case INVALID_OP_A: 	msgstr = "NaN as operand";
				   			break;

		case INVALID_OP_B:
		case INVALID_OP_B1:
		case INVALID_OP_B2: msgstr = "both operands INF";
				   			break;

		case INVALID_OP_C: 	msgstr = "0 * INF";
				   			break;

		case INVALID_OP_D1: msgstr = "0/0";
				   			break;

		case INVALID_OP_D2: msgstr = "INF/INF";
				    		break;

		case INVALID_OP_D3: msgstr = "divisor not normalized";
				    		break;

		case INVALID_OP_E1: msgstr = "<num> MOD 0";
				    		break;


		case INVALID_OP_E2: msgstr = "INF MOD <num>";
				    		break;

		case INVALID_OP_F2: msgstr = "operand INF";
				    		break;

		    
		case INVALID_OP_F3: msgstr = "operand not normalized";
				    		break;

		case INVALID_OP_G:  msgstr = "result not representable";
				    		break;

		case INVALID_OP_H:  msgstr = "unordered comparison (Nan)";
				    		break;

		case INVALID_OP_H1:  msgstr = "unordered comparison (both INF)";
				    		break;

		case DIVZERO:	    msgstr = "divide by zero";
				    		break;

		case OVERFL: 	    msgstr = "overflow";
				    		break;

		case UNDERFL:
		case UNDERFLOW_B:
		case UNDERFLOW_A:   msgstr = "underflow";
							break;	

		case CONVERT_INFINITY:
				    		msgstr = 
		    "attempt to print INF/Nan or input value overflow";
				    		break;

		case DOMAIN_ERROR:
							msgstr = "domain error";
							break;

		case CANT_REDUCE_RANGE:
							msgstr = 
			"operand too large to derive valid answer";
							break;

		case PARTIAL_SLOSS:
							msgstr = "partial loss of significance";
							break;
		case INEXACT:
				msgstr = "inexact";
				break;

		case ILLEGAL_OP:
				msgstr = "illegal operation";
				break;

		case UNKN_ERROR:
				msgstr = "unrecognizable error";
				break;

		default:	    msgstr = "!!UNKNOWN ERROR CODE";

	}

	fprintf(stderr,"%s. ",msgstr);
	if (op == MATH) {
		if (_fperror.precision == SINGLE) {
			if (op == PARTIAL_SLOSS) {
				if (!ISFMaxExp(_fperror.val.fval))
					fprintf(stderr,"result set to %g.", _fperror.val.fval);
			}
			else {
				if (!ISFMaxExp(_fptemp.fval) && 
				  ((_fperror.type & 0xff00)!=INVALID_OP))
					fprintf(stderr,"operand was %g.",_fptemp.fval);
			}
		}
		else {
			if (op == PARTIAL_SLOSS) {
				if (!ISMaxExp(_fperror.val.dval)) 
					fprintf(stderr,"result set to %lg.", _fperror.val.dval);
			}
			else {
				if ((!ISMaxExp(_fptemp.dval) &&
				  ((_fperror.type & 0xff00)!=INVALID_OP)))
					fprintf(stderr,"operand was %lg.",_fptemp.dval);
			}
		}
	}

	fprintf(stderr,"\n");

}

static
sethw(mask)
unsigned char mask;
{
	/* if this is a Turbo, and we are using the fpa, we need
	   to set the error mask register.
	*/
	unsigned long temp;
	char cputypestring[0x10];

	/* set the turbo fpa flag to FALSE */
	_has_turbo_fpa = 0;

	if (!getversion(KVERS_CPUTYPE,cputypestring))
	{
		/* call succeeded.  the buffer is either
		   "ipII" or "pmII"
		*/
		if (cputypestring[0] == 'i')  {
			/* we possibly have an fpa. */
			/* if the start location of our program is > 0x2000,
	   		   we have an fpa. */
			_has_turbo_fpa = hasfpa();
		}
	}

	if (_has_turbo_fpa) {
		/* have to set the error mask register on the Turbo fpa */
		setfpamask(mask);
		temp = (mask << 16);
		if ((temp & (IGN_DENORM|IGN_UNDERFLOW)) ==
			(IGN_DENORM|IGN_UNDERFLOW))
			/*  run in fast mode */
			setfastmode();
		else
			setieeemode();
		fpspl0();
	}
}


static
hw_to_swerror(result)
int result;
{

			if (result & IGN_DIVZERO)
				_fperror.type = DIVZERO;
			else if (result & IGN_OVERFLOW)
				_fperror.type = OVERFL;
			else if (result & IGN_UNDERFLOW)
				_fperror.type = UNDERFL;
			else if (result & IGN_DENORM)
				_fperror.type = INVALID_OP_F3;
			else if (result & IGN_NANOP)
				_fperror.type = INVALID_OP_A;
			else if (result & IGN_INEXACT)
				_fperror.type = INEXACT;
			else if (result & IGN_UNKNOWN)
				_fperror.type = UNKN_ERROR;
			else 
				_fperror.type = ILLEGAL_OP;

	_fperror.operation = 0;
}

#ifdef NOTDEF
float 
_raise_fperror(op,type)
{
	/* raise a floating point exception on the last operation */
	_fperror.type = type;
	_fperror.operation = op;
	_fperror.precision = (op == FIX)?UNKNOWN:SINGLE;
	_is_software = 1;
	if (op != MATH) _fperror.val.fval = 0.0;
	if (fakehwerror() != 0) 
		kill (getpid(),SIGFPE);
	_is_software = 0;
	return(_fperror.val.fval);
}


long float 
_lraise_fperror(op,type)

{
	/* raise a floating point exception on the last operation */
	_fperror.type = type;
	_fperror.operation = op;
	_fperror.precision = DOUBLE;
	_is_software = 1;
	if (op != MATH) _fperror.val.dval= 0.0;
	if (fakehwerror() != 0) 
		kill (getpid(),SIGFPE);
	_is_software = 0;
	return(_fperror.val.dval);
}

_UPERR(type)
{
	_is_software = 1;
	_is_fortran = 1;
	_fperror.type = type;
	_fperror.operation = 0;
	_fperror.precision = UNKNOWN;
	_fperror.val.fval = 0.0;
	if (fakehwerror() != 0) 
		kill (getpid(),SIGFPE);
	_is_software = 0;
	_is_fortran = 0;
	return(0);
}

float 
_FPERR(type)
{
	_is_software = 1;
	_is_fortran = 1;
	_fperror.type = type;
	_fperror.operation = 0;
	_fperror.precision = SINGLE;
	_fperror.val.fval = 0.0;
	if (fakehwerror() != 0) 
		kill (getpid(),SIGFPE);
	_is_software = 0;
	_is_fortran = 0;
	return(_fperror.val.fval);
}

long float 
_LFPERR(type)
{
	_is_software = 1;
	_is_fortran = 1;
	_fperror.type = type;
	_fperror.operation = 0;
	_fperror.precision = DOUBLE;
	_fperror.val.dval = 0.0;
	if (fakehwerror() != 0) 
		kill (getpid(),SIGFPE);
	_is_software = 0;
	_is_fortran = 0;
	return(_fperror.val.dval);
}
#endif

fortran setfpe(mask,options)
unsigned long *mask,*options;
{
	unsigned long derefmask,derefoptions;

	_save_fortran_registers();
	derefmask = *mask;
	derefoptions = *options;
	derefoptions |= FORT_HANDLER;
	_set_forthandler(derefoptions);
	fpsigset(0,(derefmask<<16)|derefoptions);
}
